/**
 * 快捷键映射配置
 */
const SHORTCUTS = {
    'undo': 'Ctrl+Z',
    'redo': 'Ctrl+Y',
    'save': 'Ctrl+S',
    'open': 'Ctrl+O',
    'new': 'Ctrl+N',
    'close': 'Esc',
    'cancel': 'Esc',
    'clear': 'Del',
    'delete': 'Del',
    'copy': 'Ctrl+C',
    'paste': 'Ctrl+V',
    'cut': 'Ctrl+X',
    'selectAll': 'Ctrl+A',
    'find': 'Ctrl+F',
    'replace': 'Ctrl+H',
    'fullscreen': 'F11',
    'toggle': 'Space',
    'play': 'Space',
    'pause': 'Space',
    'screenshot': 'Ctrl+Shift+S',
    'record': 'Ctrl+Shift+R',
    'pin': 'Ctrl+P',
    'pen': 'P',
    'eraser': 'E',
    'arrow': 'A',
    'rect': 'R',
    'circle': 'C',
    'text': 'T',
    'mosaic': 'M',
    'refimg': 'Ctrl+I'
};
/**
 * 工具提示增强器
 * 自动为按钮添加快捷键提示
 */
export class TooltipEnhancer {
    enhancedElements;
    constructor() {
        this.enhancedElements = new Set();
    }
    /**
     * 增强所有按钮的工具提示
     */
    enhanceAll() {
        const buttons = document.querySelectorAll('button[data-tool], button[data-rec-action], button[data-shot-action], button[data-shot-tool]');
        for (const button of buttons) {
            this.enhanceButton(button);
        }
    }
    /**
     * 增强单个按钮的工具提示
     * @param {HTMLElement} button - 按钮元素
     */
    enhanceButton(button) {
        if (this.enhancedElements.has(button))
            return;
        const tool = button.getAttribute('data-tool') ||
            button.getAttribute('data-rec-action') ||
            button.getAttribute('data-shot-action') ||
            button.getAttribute('data-shot-tool');
        if (!tool)
            return;
        const shortcut = SHORTCUTS[tool];
        if (!shortcut)
            return;
        const currentTitle = button.getAttribute('title') || button.getAttribute('aria-label') || '';
        if (currentTitle.includes('(') && currentTitle.includes(')')) {
            return;
        }
        const newTitle = currentTitle ? `${currentTitle} (${shortcut})` : shortcut;
        button.setAttribute('title', newTitle);
        button.setAttribute('data-shortcut', shortcut);
        this.enhancedElements.add(button);
    }
    /**
     * 移除所有增强
     */
    removeAll() {
        for (const element of this.enhancedElements) {
            const shortcut = element.getAttribute('data-shortcut');
            if (shortcut) {
                const title = element.getAttribute('title') || '';
                const newTitle = title.replace(` (${shortcut})`, '');
                element.setAttribute('title', newTitle);
                element.removeAttribute('data-shortcut');
            }
        }
        this.enhancedElements.clear();
    }
    /**
     * 获取工具的快捷键
     * @param {string} tool - 工具名称
     * @returns {string|null} 快捷键字符串
     */
    getShortcut(tool) {
        return SHORTCUTS[tool] || null;
    }
    /**
     * 设置工具的快捷键
     * @param {string} tool - 工具名称
     * @param {string} shortcut - 快捷键字符串
     */
    setShortcut(tool, shortcut) {
        SHORTCUTS[tool] = shortcut;
    }
}
/**
 * 创建工具提示增强器实例
 * @returns {TooltipEnhancer} 工具提示增强器实例
 */
export function createTooltipEnhancer() {
    return new TooltipEnhancer();
}
export default createTooltipEnhancer();
//# sourceMappingURL=tooltipEnhancer.js.map