export class ToastManager {
    container;
    toastQueue;
    activeToast;
    animationFrameId;
    constructor() {
        this.container = null;
        this.toastQueue = [];
        this.activeToast = null;
        this.animationFrameId = null;
        this.init();
    }
    init() {
        this.ensureContainer();
    }
    ensureContainer() {
        if (this.container)
            return;
        this.container = document.createElement('div');
        this.container.id = 'toast-container';
        this.container.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      z-index: 9999;
      display: flex;
      flex-direction: column;
      gap: 10px;
      pointer-events: none;
    `;
        document.body.appendChild(this.container);
    }
    /**
     * 显示一个 Toast 通知
     * @param {string} message - 要显示的消息文本
     * @param {Object} options - 配置选项
     * @param {string} [options.type='info'] - 通知类型: 'info', 'success', 'warning', 'error'
     * @param {number} [options.duration=3000] - 显示持续时间（毫秒）
     * @param {string} [options.icon] - 自定义图标
     */
    show(message, options = {}) {
        const { type = 'info', duration = 3000, icon = null } = options;
        const toast = this.createToast(message, type, icon);
        this.toastQueue.push({ toast, duration, createdAt: Date.now() });
        if (!this.animationFrameId) {
            this.animationFrameId = requestAnimationFrame(this.processQueue.bind(this));
        }
    }
    createToast(message, type, icon) {
        const toast = document.createElement('div');
        toast.className = `toast toast-${type}`;
        const colors = {
            info: 'rgba(59, 130, 246, 0.95)',
            success: 'rgba(34, 197, 94, 0.95)',
            warning: 'rgba(251, 146, 60, 0.95)',
            error: 'rgba(239, 68, 68, 0.95)'
        };
        const icons = {
            info: 'ℹ️',
            success: '✓',
            warning: '⚠',
            error: '✕'
        };
        toast.style.cssText = `
      background: ${colors[type]};
      color: white;
      padding: 12px 16px;
      border-radius: 8px;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
      font-size: 14px;
      font-weight: 500;
      display: flex;
      align-items: center;
      gap: 10px;
      min-width: 200px;
      max-width: 400px;
      opacity: 0;
      transform: translateX(20px);
      transition: opacity 0.2s ease, transform 0.2s ease;
    `;
        const iconElement = document.createElement('span');
        iconElement.textContent = icon || icons[type];
        iconElement.style.fontSize = '18px';
        const textElement = document.createElement('span');
        textElement.textContent = message;
        textElement.style.flex = '1';
        toast.appendChild(iconElement);
        toast.appendChild(textElement);
        return toast;
    }
    processQueue(_timestamp) {
        this.ensureContainer();
        if (!this.container) {
            this.animationFrameId = null;
            return;
        }
        if (this.toastQueue.length === 0) {
            this.animationFrameId = null;
            return;
        }
        const now = Date.now();
        if (!this.activeToast) {
            const next = this.toastQueue.shift();
            if (!next) {
                this.animationFrameId = null;
                return;
            }
            const { toast, duration } = next;
            this.activeToast = { toast, duration, createdAt: now };
            this.container.appendChild(toast);
            requestAnimationFrame(() => {
                toast.style.opacity = '1';
                toast.style.transform = 'translateX(0)';
            });
        }
        else {
            const { toast, duration, createdAt } = this.activeToast;
            const elapsed = now - createdAt;
            if (elapsed >= duration) {
                this.hideActiveToast();
            }
        }
        if (this.toastQueue.length > 0 || this.activeToast) {
            this.animationFrameId = requestAnimationFrame(this.processQueue.bind(this));
        }
    }
    hideActiveToast() {
        if (!this.activeToast)
            return;
        const { toast } = this.activeToast;
        toast.style.opacity = '0';
        toast.style.transform = 'translateX(20px)';
        setTimeout(() => {
            if (toast.parentNode) {
                toast.parentNode.removeChild(toast);
            }
            this.activeToast = null;
        }, 200);
    }
    /**
     * 显示信息类型的通知
     * @param {string} message - 要显示的消息
     * @param {Object} [options] - 配置选项
     */
    info(message, options) {
        this.show(message, { ...options, type: 'info' });
    }
    /**
     * 显示成功类型的通知
     * @param {string} message - 要显示的消息
     * @param {Object} [options] - 配置选项
     */
    success(message, options) {
        this.show(message, { ...options, type: 'success' });
    }
    /**
     * 显示警告类型的通知
     * @param {string} message - 要显示的消息
     * @param {Object} [options] - 配置选项
     */
    warning(message, options) {
        this.show(message, { ...options, type: 'warning' });
    }
    /**
     * 显示错误类型的通知
     * @param {string} message - 要显示的消息
     * @param {Object} [options] - 配置选项
     */
    error(message, options) {
        this.show(message, { ...options, type: 'error' });
    }
    /**
     * 清除所有待显示和当前显示的通知
     */
    clear() {
        this.toastQueue = [];
        this.hideActiveToast();
        if (!this.container)
            return;
        while (this.container.firstChild) {
            this.container.removeChild(this.container.firstChild);
        }
    }
    /**
     * 销毁 Toast 管理器，清理所有资源
     */
    destroy() {
        this.clear();
        if (this.animationFrameId !== null) {
            cancelAnimationFrame(this.animationFrameId);
            this.animationFrameId = null;
        }
        if (this.container && this.container.parentNode) {
            this.container.parentNode.removeChild(this.container);
        }
        this.container = null;
    }
}
/**
 * 创建一个新的 Toast 管理器实例
 * @returns {ToastManager} Toast 管理器实例
 */
export function createToastManager() {
    return new ToastManager();
}
export default createToastManager();
//# sourceMappingURL=toastManager.js.map