export function throttle(fn, delay) {
    let lastCall = 0;
    let timeoutId = null;
    return function throttled(...args) {
        const now = Date.now();
        const timeSinceLastCall = now - lastCall;
        if (timeSinceLastCall >= delay) {
            lastCall = now;
            return fn.apply(this, args);
        }
        if (timeoutId) {
            clearTimeout(timeoutId);
        }
        timeoutId = setTimeout(() => {
            lastCall = Date.now();
            timeoutId = null;
            fn.apply(this, args);
        }, delay - timeSinceLastCall);
    };
}
export function debounce(fn, delay) {
    let timeoutId = null;
    return function debounced(...args) {
        if (timeoutId) {
            clearTimeout(timeoutId);
        }
        timeoutId = setTimeout(() => {
            timeoutId = null;
            fn.apply(this, args);
        }, delay);
    };
}
export function rafThrottle(fn) {
    let rafId = null;
    let lastArgs = null;
    return function rafThrottled(...args) {
        lastArgs = args;
        if (rafId === null) {
            rafId = requestAnimationFrame(() => {
                rafId = null;
                fn.apply(this, lastArgs);
            });
        }
    };
}
export function createThrottleManager() {
    const throttles = new Map();
    return {
        throttle(key, fn, delay) {
            if (throttles.has(key)) {
                throttles.get(key)();
            }
            const throttledFn = throttle(fn, delay);
            throttles.set(key, () => {
                throttles.clear();
            });
            return throttledFn;
        },
        clear(key) {
            if (throttles.has(key)) {
                const cancel = throttles.get(key);
                cancel();
                throttles.delete(key);
            }
        },
        clearAll() {
            for (const cancel of throttles.values()) {
                cancel();
            }
            throttles.clear();
        }
    };
}
//# sourceMappingURL=throttle.js.map