export class BroadcastManager {
    pendingUpdates;
    broadcastScheduled;
    debounceDelay;
    lastBroadcastTime;
    minBroadcastInterval;
    constructor() {
        this.pendingUpdates = new Map();
        this.broadcastScheduled = false;
        this.debounceDelay = 100;
        this.lastBroadcastTime = 0;
        this.minBroadcastInterval = 50;
    }
    queueUpdate(key, data) {
        this.pendingUpdates.set(key, {
            data,
            timestamp: Date.now()
        });
        if (!this.broadcastScheduled) {
            this.scheduleBroadcast();
        }
    }
    scheduleBroadcast() {
        this.broadcastScheduled = true;
        setTimeout(() => {
            this.broadcastScheduled = false;
            this.flushUpdates();
        }, this.debounceDelay);
    }
    flushUpdates() {
        if (this.pendingUpdates.size === 0)
            return;
        const now = Date.now();
        const timeSinceLastBroadcast = now - this.lastBroadcastTime;
        if (timeSinceLastBroadcast < this.minBroadcastInterval) {
            const delay = this.minBroadcastInterval - timeSinceLastBroadcast;
            setTimeout(() => this.flushUpdates(), delay);
            return;
        }
        const updates = {};
        for (const [key, { data }] of this.pendingUpdates) {
            updates[key] = data;
        }
        this.pendingUpdates.clear();
        this.lastBroadcastTime = now;
        return updates;
    }
    hasPendingUpdates() {
        return this.pendingUpdates.size > 0;
    }
    clearPendingUpdates() {
        this.pendingUpdates.clear();
    }
    setDebounceDelay(delay) {
        this.debounceDelay = Math.max(0, delay);
    }
    setMinBroadcastInterval(interval) {
        this.minBroadcastInterval = Math.max(0, interval);
    }
}
export default BroadcastManager;
//# sourceMappingURL=broadcastManager.js.map