export function createMouseDot() {
    let mouseDot = null;
    let trailDots = [];
    let trailPoints = [];
    let mouseDotEnabled = true;
    let moveEnabled = true;
    let clickEnabled = true;
    let styleMode = 'fixed';
    let mouseDotShowTimer = null;
    let trailCount = 6;
    let trailLerp = 0.35;
    let lastTrailConfigRead = 0;
    function ensureMouseDot() {
        if (!mouseDot) {
            mouseDot = document.createElement('div');
            mouseDot.className = 'mouse-dot';
            mouseDot.style.display = 'none';
            document.body.appendChild(mouseDot);
        }
        if (styleMode === 'trail' || styleMode === 'nebula')
            syncTrailDots();
    }
    function readTrailConfig() {
        const now = Date.now();
        if (now - lastTrailConfigRead < 200)
            return;
        lastTrailConfigRead = now;
        let nextCount = trailCount;
        let nextLerp = trailLerp;
        try {
            const cs = getComputedStyle(document.documentElement);
            const countRaw = String(cs.getPropertyValue('--mouse-trail-count') || '').trim();
            const lerpRaw = String(cs.getPropertyValue('--mouse-trail-lag') || '').trim();
            const n = Number(countRaw);
            const l = Number(lerpRaw);
            if (Number.isFinite(n))
                nextCount = Math.round(n);
            if (Number.isFinite(l))
                nextLerp = l;
        }
        catch { }
        nextCount = Math.max(0, Math.min(12, nextCount));
        if (styleMode === 'nebula' && nextCount <= 0)
            nextCount = 6;
        nextLerp = Math.max(0.05, Math.min(0.95, nextLerp));
        if (nextCount !== trailCount) {
            trailCount = nextCount;
            syncTrailDots(true);
        }
        trailLerp = nextLerp;
    }
    function syncTrailDots(resetPoints) {
        if (trailDots.length > trailCount) {
            const removing = trailDots.splice(trailCount);
            for (const d of removing) {
                try {
                    d.remove();
                }
                catch { }
            }
        }
        if (trailDots.length < trailCount) {
            for (let i = trailDots.length; i < trailCount; i += 1) {
                const dot = document.createElement('div');
                dot.className = 'mouse-trail';
                dot.style.display = 'none';
                document.body.appendChild(dot);
                trailDots.push(dot);
            }
        }
        const isNebula = styleMode === 'nebula';
        if (mouseDot) {
            if (isNebula)
                mouseDot.classList.add('nebula');
            else
                mouseDot.classList.remove('nebula');
        }
        for (let i = 0; i < trailDots.length; i += 1) {
            const dot = trailDots[i];
            if (isNebula)
                dot.classList.add('nebula');
            else
                dot.classList.remove('nebula');
            const max = Math.max(1, trailDots.length);
            const alpha = 0.48 - (i / max) * 0.32;
            dot.style.opacity = String(Math.max(0.08, alpha));
        }
        if (resetPoints)
            trailPoints = [];
    }
    function update(pos) {
        ensureMouseDot();
        if (!mouseDot)
            return;
        if (!pos || typeof pos.x !== 'number' || typeof pos.y !== 'number')
            return;
        mouseDot.style.left = pos.x + 'px';
        mouseDot.style.top = pos.y + 'px';
        if (!mouseDotEnabled || !moveEnabled) {
            for (const d of trailDots)
                d.style.display = 'none';
            return;
        }
        mouseDot.style.display = 'block';
        if (styleMode === 'trail' || styleMode === 'nebula') {
            readTrailConfig();
            if (trailCount <= 0) {
                for (const d of trailDots)
                    d.style.display = 'none';
            }
            else {
                for (let i = 0; i < trailDots.length; i += 1) {
                    const prev = i === 0 ? pos : (trailPoints[i - 1] || pos);
                    const cur = trailPoints[i] || { x: pos.x, y: pos.y };
                    const nextX = cur.x + (prev.x - cur.x) * trailLerp;
                    const nextY = cur.y + (prev.y - cur.y) * trailLerp;
                    trailPoints[i] = { x: nextX, y: nextY };
                    const dot = trailDots[i];
                    dot.style.left = nextX + 'px';
                    dot.style.top = nextY + 'px';
                    dot.style.display = 'block';
                }
            }
        }
        else {
            for (const d of trailDots)
                d.style.display = 'none';
        }
        const visibleFor = 200;
        if (mouseDotShowTimer)
            clearTimeout(mouseDotShowTimer);
        mouseDotShowTimer = window.setTimeout(() => {
            if (!mouseDotEnabled)
                return;
            if (mouseDot)
                mouseDot.style.display = 'none';
            for (const d of trailDots)
                d.style.display = 'none';
        }, visibleFor);
    }
    function setStyleMode(mode) {
        const m = String(mode || '').trim().toLowerCase();
        const next = m === 'trail' || m === 'nebula' ? m : 'fixed';
        if (styleMode === next)
            return;
        styleMode = next;
        lastTrailConfigRead = 0;
        readTrailConfig();
        syncTrailDots(true);
        if (styleMode !== 'trail' && styleMode !== 'nebula') {
            for (const d of trailDots) {
                d.style.display = 'none';
                d.remove();
            }
            trailDots = [];
            trailPoints = [];
        }
    }
    function setVisible(visible) {
        mouseDotEnabled = !!visible;
        ensureMouseDot();
        if (!mouseDot)
            return;
        if (!mouseDotEnabled) {
            mouseDot.style.display = 'none';
            for (const d of trailDots)
                d.style.display = 'none';
            if (mouseDotShowTimer) {
                clearTimeout(mouseDotShowTimer);
                mouseDotShowTimer = null;
            }
        }
    }
    function setMoveVisible(visible) {
        moveEnabled = !!visible;
        ensureMouseDot();
        if (!mouseDot)
            return;
        if (!moveEnabled && !clickEnabled) {
            mouseDot.style.display = 'none';
            for (const d of trailDots)
                d.style.display = 'none';
            if (mouseDotShowTimer) {
                clearTimeout(mouseDotShowTimer);
                mouseDotShowTimer = null;
            }
        }
    }
    function setClickVisible(visible) {
        clickEnabled = !!visible;
        ensureMouseDot();
        if (!mouseDot)
            return;
        if (!clickEnabled && !moveEnabled) {
            mouseDot.style.display = 'none';
            for (const d of trailDots)
                d.style.display = 'none';
            if (mouseDotShowTimer) {
                clearTimeout(mouseDotShowTimer);
                mouseDotShowTimer = null;
            }
        }
    }
    function pulseClick(payload) {
        ensureMouseDot();
        if (!mouseDot)
            return;
        if (mouseDotEnabled && clickEnabled) {
            const p = payload && payload.pos;
            if (p && typeof p.x === 'number' && typeof p.y === 'number') {
                mouseDot.style.left = p.x + 'px';
                mouseDot.style.top = p.y + 'px';
            }
            mouseDot.style.display = 'block';
            mouseDot.classList.remove('mouse-click');
            void mouseDot.offsetWidth;
            mouseDot.classList.add('mouse-click');
            const visibleFor = 250;
            if (mouseDotShowTimer)
                clearTimeout(mouseDotShowTimer);
            mouseDotShowTimer = window.setTimeout(() => { if (!mouseDotEnabled)
                return; if (mouseDot)
                mouseDot.style.display = 'none'; }, visibleFor);
        }
    }
    return { update, setVisible, setMoveVisible, setClickVisible, setStyleMode, pulseClick };
}
//# sourceMappingURL=mouse.js.map