export function createKeyBubble({ container, getFadeTime }) {
    let keyBubbleEl = null;
    let keyBubbleTimer = null;
    let lastBubbleId = null;
    let lastBubbleCount = 0;
    let lastCopyText = '';
    function ensureEl() {
        if (!keyBubbleEl) {
            keyBubbleEl = document.createElement('div');
            keyBubbleEl.className = 'key-bubble';
            container.appendChild(keyBubbleEl);
            keyBubbleEl.addEventListener('click', async (e) => {
                const t = e && e.target ? e.target : null;
                if (!t || !t.classList || !t.classList.contains('kb-copy'))
                    return;
                const text = String(lastCopyText || '');
                if (!text)
                    return;
                let ok = false;
                try {
                    if (navigator && navigator.clipboard && typeof navigator.clipboard.writeText === 'function') {
                        await navigator.clipboard.writeText(text);
                        ok = true;
                    }
                }
                catch { }
                if (!ok) {
                    try {
                        const ta = document.createElement('textarea');
                        ta.value = text;
                        ta.style.position = 'fixed';
                        ta.style.left = '-9999px';
                        ta.style.top = '0';
                        document.body.appendChild(ta);
                        ta.focus();
                        ta.select();
                        ok = !!document.execCommand('copy');
                        ta.remove();
                    }
                    catch { }
                }
                showText(ok ? '已复制详情' : '复制失败');
            });
        }
    }
    function scheduleFade() {
        const t = typeof getFadeTime === 'function' ? Number(getFadeTime()) || 1000 : 1000;
        if (keyBubbleTimer)
            clearTimeout(keyBubbleTimer);
        keyBubbleTimer = window.setTimeout(() => { keyBubbleEl && keyBubbleEl.classList.add('fade'); }, t);
    }
    function showText(text) {
        ensureEl();
        if (!keyBubbleEl)
            return;
        keyBubbleEl.textContent = text;
        if (keyBubbleTimer)
            clearTimeout(keyBubbleTimer);
        keyBubbleEl.classList.remove('fade');
        void keyBubbleEl.offsetWidth;
        scheduleFade();
    }
    function showToast(opts) {
        const o = (opts && typeof opts === 'object') ? opts : {};
        const type = String(o.type || '').trim().toLowerCase();
        const prefix = type === 'error' ? '错误' : type === 'success' ? '完成' : type === 'warn' ? '提示' : '提示';
        const msg = String(o.message || '').trim();
        const rawDetails = String(o.details || '').trim();
        lastCopyText = rawDetails;
        if (!rawDetails) {
            const out = msg ? `${prefix}：${msg}` : prefix;
            showText(out);
            return;
        }
        let details = rawDetails;
        const lines = details.split(/\r?\n/).map(x => String(x || '').trim()).filter(Boolean).slice(0, 3);
        details = lines.join('\n');
        if (details.length > 280)
            details = `${details.slice(0, 280)}…`;
        const html = `<span class="kb-text">${prefix}：${escapeHtml(msg || prefix)}</span><button class="kb-copy" type="button">复制详情</button>`;
        const body = details ? `${html}<span class="kb-text">${escapeHtml(details)}</span>` : html;
        showHtml(body, `${prefix}:${msg}`);
    }
    function escapeHtml(s) {
        return String(s || '').replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#39;');
    }
    function showHtml(html, keyId) {
        ensureEl();
        if (!keyBubbleEl)
            return;
        if (keyId && lastBubbleId === keyId && !keyBubbleEl.classList.contains('fade')) {
            lastBubbleCount += 1;
        }
        else {
            lastBubbleId = keyId || null;
            lastBubbleCount = 1;
        }
        const content = `<span class="kb-wrap">${html}${lastBubbleCount > 1 ? `<span class=\"kb-count\">×${lastBubbleCount}</span>` : ''}</span>`;
        keyBubbleEl.innerHTML = content;
        if (keyBubbleTimer)
            clearTimeout(keyBubbleTimer);
        keyBubbleEl.classList.remove('fade');
        void keyBubbleEl.offsetWidth;
        scheduleFade();
    }
    return { showText, showToast, showHtml };
}
//# sourceMappingURL=bubble.js.map